#!/bin/bash
######################################################################
# Melissa Data Address Object Java interface for Unix build script
######################################################################
# Use this script to create the shared object library file and the jar
# file for the Address Object Java interface.  Once those have been
# created, you can then run RunSample.sh to see how it may be used.
#
# Usage: ./BuildWrapper.sh [<OUTPUT_DIRECTORY>]
#
# Takes one optional parameter, the path name of the directory where
# the object binary files will be placed once they are built.  If no
# parameter is given, we use the current directory.
#
# We also need to know where the Melissa Data Address Object component
# was installed.  This should be the pathname of the "AddrObj"
# subdirectory, which is created in a default install of Address
# Object.  If the environment variable "MDENV_ADDROBJ_PATH" is
# defined, we use that.  If that is not defined, we fall back to a
# hard coded value, which should be changed to be useful to the local
# environment.  The provided value is expected to fail if it is not
# otherwise overridden.
#
# To compile the C++ parts of the interface, we just use whatever the
# name "g++" resolves to in the current environment, presuming that
# will be the preferred compilation environment.  To compile the Java
# part of the interface, we must allow for different Java
# environments.  If "JAVA_HOME" is defined, then we base all of the
# needed paths on that.
#
# If that is not defined in the environment, then we search for the
# Java compiler and jar tool using which(1), so that we get a full
# path name for them.  If the "MDENV_JAVA_INCLUDE" variable is set,
# then we use that to find the Java include files required to compile
# the C++ part of the interface.  If it is not set, we try a few 
# common places related to the path of the Java compiler.
######################################################################
set -u
declare -r SCRIPT_NAME=$(basename $0 .sh)

# where are the mdAddr files located?
if [ "${MDENV_ADDROBJ_PATH:+DEFINED}" == 'DEFINED' ]; then
    # if the environment variable is set, just use that
    declare -r mdAddrPath="${MDENV_ADDROBJ_PATH}"
else
    ###EDIT_THIS##  use a hard coded default path specific to the local environment
    declare -r mdAddrPath="The Path To Your AddrObj Directory"
fi

# standard error message and exit function
function errexit ()
{
    echo ""  1>&2
    echo "${SCRIPT_NAME}: ERROR: ${*}" 1>&2
    exit 1
}

# standard warning message function
function warn ()
{
    echo ""  1>&2
    echo "${SCRIPT_NAME}: Warning: ${*}"  1>&2
}

# define the conditions a valid file must meet
function is_valid_file ()
{
    if [ ! "${1}" ]; then
        warn "null name given to is_valid_file"
        return 0
    fi

    if [ ! -e "${1}" ]; then errexit "does not exist: ${1}"; fi

    if [ ! -f "${1}" ]; then errexit "is not a regular file: ${1}"; fi

    if [ ! -r "${1}" ]; then errexit "lacks read permission: ${1}"; fi

    return 0
}


# define the conditions a valid executable file must meet
function is_valid_exec ()
{
    is_valid_file "${1}"

    if [ ! -x "${1}" ]; then errexit "lacks execute permission: ${1}"; fi

    return 0
}

# define the conditions a valid directory must meet
function is_dir_readable ()
{
    if [ ! "${1}" ]; then
        warn "null name given to is_dir_readable"
        return 0
    fi

    if [ ! -e "${1}" ]; then errexit "does not exist: ${1}"; fi

    if [ ! -d "${1}" ]; then errexit "is not a directory: ${1}"; fi

    if [ ! -r "${1}" ]; then errexit "lacks read permission: ${1}"; fi

    if [ ! -x "${1}" ]; then errexit "lacks traverse permission: ${1}"; fi

    return 0
}

function is_dir_writable ()
{
    is_dir_readable "${1}"

    if [ ! -w "${1}" ]; then errexit "lacks write permission: ${1}"; fi

    return 0
}

declare -r JWRAP_SRC='mdAddrJavaWrapper.cpp'
declare -r JWRAP_OBJ='mdAddrJavaWrapper.o'
declare -r JWRAP_SO='libmdAddrJavaWrapper.so'
declare -r JWRAP_JAR='mdAddr.jar'

# decide where the outputs need to go
if [ $# -eq 0 ]; then
    # just leave them in the current directory if we get no parameter
    declare -r OUTPUT='.'
else
    # we only are expecting one parameter at most
    if [ $# -gt 1 ]; then
        warn "excess parameters were supplied: one is the maximum, $# were seen"
    fi
    declare -r OUTPUT="${1}"
fi

echo ""
echo "Starting build for Linux Java interface for mdAddr object"
echo ""
# ensure some basic things are true
is_dir_writable "${OUTPUT}"
is_dir_readable 'com'
is_dir_writable 'com/melissadata'

# make sure the given AddrObj directory contains the required items
is_valid_file "${mdAddrPath}/libmdAddr.so"
is_valid_file "${mdAddrPath}/mdAddr.h"
is_valid_file "${mdAddrPath}/mdEnums.h"

# we need to find the right Java environment to use, it is a little tricky

# First, if we do have a JAVA_HOME environment variable defined, we use that
if [ "${JAVA_HOME:+DEFINED}" == 'DEFINED' ]; then
    # JAVA_HOME is assumed to be the root of all Java related paths 
    declare -r JavaDir_1="${JAVA_HOME}/include"
    is_dir_readable "${JavaDir_1}"

    declare -r JavaDir_2="${JavaDir_1}/linux"
    is_dir_readable "${JavaDir_2}"

    is_dir_readable "${JAVA_HOME}/bin"
    declare -r JAVAC_CMD="${JAVA_HOME}/bin/javac"
    is_valid_exec "${JAVAC_CMD}"
    declare -r JAR_CMD="${JAVA_HOME}/bin/jar"
    is_valid_exec "${JAR_CMD}"
else
    # see if there is a 'javac' anywhere in the path set
    if tmp=$(which javac)
    then
        declare -r JAVAC_CMD="${tmp}"
        is_valid_exec "${JAVAC_CMD}"
    else
        errexit "no instance of the 'javac' command could be found"
    fi

    # see if there is a 'jar' anywhere in the path set
    if tmp=$(which jar)
    then
        declare -r JAR_CMD="${tmp}"
        is_valid_exec "${JAR_CMD}"
    else
        errexit "no instance of the 'jar' command could be found"
    fi

    # now we need the include directory, is it in the environment?
    if [ "${MDENV_JAVA_INCLUDE:+DEFINED}" == 'DEFINED' ]; then
        if [ -d "${MDENV_JAVA_INCLUDE}" \
             -a -d "${MDENV_JAVA_INCLUDE}/linux" ]
        then
            declare -r JavaDir_1="${MDENV_JAVA_INCLUDE}"
            is_dir_readable "${JavaDir_1}"

            declare -r JavaDir_2="${JavaDir_1}/linux"
            is_dir_readable "${JavaDir_2}"
        else
            errexit "the header files (include directory) for Java were not found"
        fi
    else
        # try to find the includes in the ancestors of the javac command we found
        tmp=$(dirname "${JAVAC_CMD}")
        case ${tmp} in
            */bin ) 
                # this is promising, see if the grandparent has the includes
                if [ -d $(dirname "${tmp}")'/include' \
                     -a -d $(dirname "${tmp}")'/include/linux' ]
                then
                    declare -r JavaDir_1=$(dirname "${tmp}")'/include'
                    is_dir_readable "${JavaDir_1}"
                    declare -r JavaDir_2="${JavaDir_1}/linux"
                    is_dir_readable "${JavaDir_2}"
                elif [ -d $(dirname $(dirname "${tmp}"))'/include' \
                       -a -d $(dirname $(dirname "${tmp}"))'/include/linux' ]
                then
                    # great grandparent might be the one, try it
                    declare -r JavaDir_1=$(dirname $(dirname "${tmp}"))'/include'
                    is_dir_readable "${JavaDir_1}"
                    declare -r JavaDir_2="${JavaDir_1}/linux"
                    is_dir_readable "${JavaDir_2}"
                else
                    errexit "the header files (include directory) for Java were not found"
                fi
                ;;

            * ) 
                # just see if the grandparent has the includes
                if [ -d $(dirname "${tmp}")'/include'  \
                     -a -d $(dirname "${tmp}")'/include/linux' ]
                then
                    declare -r JavaDir_1=$(dirname "${tmp}")'/include'
                    is_dir_readable "${JavaDir_1}"
                    declare -r JavaDir_2="${JavaDir_1}/linux"
                    is_dir_readable "${JavaDir_2}"
                else
                    errexit "the header files (include directory) for Java were not found"
                fi
                ;;
        esac
    fi
fi

# now we know where things are, put it all together
echo "Output Directory: ${OUTPUT}"
echo "    AddrObj Path: ${mdAddrPath}"
echo "   Java compiler: ${JAVAC_CMD}"
echo "   Java includes: ${JavaDir_1}"
echo ""

# compile the wrapper library code
echo "Compiling Java interface library now..."
if ! g++ -fpic -c -I"${mdAddrPath}" -I"${JavaDir_1}" -I"${JavaDir_2}" "${JWRAP_SRC}"
then
    errexit "unable to compile Java wrapper object"
fi

# and then link it into a shared object along with our mdAddr library
echo "Linking Java interface shared library now..."
if ! g++ -g -shared "${JWRAP_OBJ}" -o "${OUTPUT}/${JWRAP_SO}" -L"${mdAddrPath}" -lmdAddr -pthread
then
    errexit "unable to link Java wrapper shared object"
fi

# no need for the intermediate files now
rm -f "${JWRAP_OBJ}"

# go to where the java files live
if ! pushd 'com/melissadata' >/dev/null 2>&1
then
    errexit "unable to pushd to ${PWD}/com/melissadata"
fi

# compile them into the class files needed
echo "Compiling Java classes now..."
if ! ${JAVAC_CMD} *.java
then
    errexit "unable to build Java classes for wrapper"
fi

# back to where we began
popd >/dev/null 2>&1

# and jar up the classes into our final interface bundle
echo "Creating jar file now..."
if ! ${JAR_CMD} cf "${OUTPUT}/${JWRAP_JAR}" 'com'
then
    errexit "unable to build jar file for wrapper"
fi

# then clean up the class files, no need once we have the jar
find '.' -name '*.class' -exec rm \{} \;

echo ""
echo "All done, please use RunSample.sh to see a brief demonstration."
echo ""

exit 0
